// This code is a part of the Telephony Framework C++ Library.
// Copyright (C) 1997 Chris Sells. All rights reserved.
// tTranslateCaps.h: CtTranslateCaps class interface

#ifndef TTRANSLATECAPS_H
#define TTRANSLATECAPS_H

#include "tVarData.h"

class CtTranslateCaps : public CtVariableData
{
public:
    operator const LPLINETRANSLATECAPS() const;

    TRESULT GetTranslateCaps();

    DWORD   GetCurrentLocationID() const;
    DWORD   GetNumLocations() const;

    DWORD   GetPermanentLocationID(DWORD nLocation) const;
    LPCSTR  GetLocationName(DWORD nLocation) const;
    DWORD   GetCountryCode(DWORD nLocation) const;
    LPCSTR  GetAreaCode(DWORD nLocation) const; // Forwards to GetCityCode()
    LPCSTR  GetCityCode(DWORD nLocation) const;
    DWORD   GetPreferredCardID(DWORD nLocation) const;
    LPCSTR  GetLocalAccessCode(DWORD nLocation) const;
    LPCSTR  GetLongDistanceAccessCode(DWORD nLocation) const;
    LPCSTR  GetTollPrefixList(DWORD nLocation) const;
    DWORD   GetCountryID(DWORD nLocation) const;
    DWORD   GetLocationOptions(DWORD nLocation) const;
    LPCSTR  GetCancelCallWaiting(DWORD nLocation) const;

    DWORD   GetCurrentPreferredCardID() const;
    DWORD   GetNumCards() const;

    DWORD   GetPermanentCardID(DWORD nCard) const;
    LPCSTR  GetCardName(DWORD nCard) const;
    DWORD   GetCardNumberDigits(DWORD nCard) const;
    LPCSTR  GetSameAreaRule(DWORD nCard) const;
    LPCSTR  GetLongDistanceRule(DWORD nCard) const;
    LPCSTR  GetInternationalRule(DWORD nCard) const;
    DWORD   GetCardOptions(DWORD nCard) const;

protected:
    virtual TRESULT FillBuffer();

private:
    const LPLINETRANSLATECAPS GetData() const;
};

// Inline member functions: For Compiler Eyes Only!

inline
CtTranslateCaps::operator const LPLINETRANSLATECAPS() const
{
    return LPLINETRANSLATECAPS(m_pd);
}

inline
const LPLINETRANSLATECAPS CtTranslateCaps::GetData() const
{
    assert(m_pd);
    return LPLINETRANSLATECAPS(m_pd);
}

inline
TRESULT CtTranslateCaps::GetTranslateCaps()
{
    assert(CtLine::GetAppHandle());

    return UpdateData();
}

inline
TRESULT CtTranslateCaps::FillBuffer()
{
    assert(m_pd->dwTotalSize >= sizeof(LINETRANSLATECAPS));
    return ::TfxTapiFunc(::lineGetTranslateCaps(CtLine::GetAppHandle(),
                                                CtLine::GetAppVersion(),
                                                LPLINETRANSLATECAPS(m_pd)),
                         TDBSTR("lineGetTranslateCaps"));
}

inline
DWORD CtTranslateCaps::GetCurrentLocationID() const
{
    return GetData()->dwCurrentLocationID;
}

inline
DWORD CtTranslateCaps::GetNumLocations() const
{
    return GetData()->dwNumLocations;
}

inline
DWORD CtTranslateCaps::GetPermanentLocationID(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return alle[nLocation].dwPermanentLocationID;
}

inline
LPCSTR CtTranslateCaps::GetLocationName(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return GetStringPtr(alle[nLocation].dwLocationNameOffset,
                        alle[nLocation].dwLocationNameSize);
}

inline
DWORD CtTranslateCaps::GetCountryCode(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return alle[nLocation].dwCountryCode;
}

inline
LPCSTR CtTranslateCaps::GetAreaCode(DWORD nLocation) const
{
    return GetCityCode(nLocation);
}

inline
LPCSTR CtTranslateCaps::GetCityCode(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return GetStringPtr(alle[nLocation].dwCityCodeOffset,
                        alle[nLocation].dwCityCodeSize);
}

inline
DWORD CtTranslateCaps::GetPreferredCardID(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return alle[nLocation].dwPreferredCardID;
}

inline
LPCSTR CtTranslateCaps::GetLocalAccessCode(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return GetStringPtr(alle[nLocation].dwLocalAccessCodeOffset,
                        alle[nLocation].dwLocalAccessCodeSize);
}

inline
LPCSTR CtTranslateCaps::GetLongDistanceAccessCode(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return GetStringPtr(alle[nLocation].dwLongDistanceAccessCodeOffset,
                        alle[nLocation].dwLongDistanceAccessCodeSize);
}

inline
LPCSTR CtTranslateCaps::GetTollPrefixList(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return GetStringPtr(alle[nLocation].dwTollPrefixListOffset,
                        alle[nLocation].dwTollPrefixListSize);
}

inline
DWORD CtTranslateCaps::GetCountryID(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return alle[nLocation].dwCountryID;
}

inline
DWORD CtTranslateCaps::GetLocationOptions(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return alle[nLocation].dwOptions;
}

inline
LPCSTR CtTranslateCaps::GetCancelCallWaiting(DWORD nLocation) const
{
    assert(nLocation < GetNumLocations());
    LPLINELOCATIONENTRY   alle = (LPLINELOCATIONENTRY)GetDataPtr(GetData()->dwLocationListOffset);
    assert(alle);

    return GetStringPtr(alle[nLocation].dwCancelCallWaitingOffset,
                        alle[nLocation].dwCancelCallWaitingSize);
}

inline
DWORD CtTranslateCaps::GetCurrentPreferredCardID() const
{
    return GetData()->dwCurrentPreferredCardID;
}

inline
DWORD CtTranslateCaps::GetNumCards() const
{
    return GetData()->dwNumCards;
}

inline
DWORD CtTranslateCaps::GetPermanentCardID(DWORD nCard) const
{
    assert(nCard < GetNumCards());
    LPLINECARDENTRY alce = (LPLINECARDENTRY)GetDataPtr(GetData()->dwCardListOffset);
    assert(alce);

    return alce[nCard].dwPermanentCardID;
}

inline
LPCSTR CtTranslateCaps::GetCardName(DWORD nCard) const
{
    assert(nCard < GetNumCards());
    LPLINECARDENTRY alce = (LPLINECARDENTRY)GetDataPtr(GetData()->dwCardListOffset);
    assert(alce);

    return GetStringPtr(alce[nCard].dwCardNameOffset,
                        alce[nCard].dwCardNameSize);
}

inline
DWORD CtTranslateCaps::GetCardNumberDigits(DWORD nCard) const
{
    assert(nCard < GetNumCards());
    LPLINECARDENTRY alce = (LPLINECARDENTRY)GetDataPtr(GetData()->dwCardListOffset);
    assert(alce);

    return alce[nCard].dwCardNumberDigits;
}

inline
LPCSTR CtTranslateCaps::GetSameAreaRule(DWORD nCard) const
{
    assert(nCard < GetNumCards());
    LPLINECARDENTRY alce = (LPLINECARDENTRY)GetDataPtr(GetData()->dwCardListOffset);
    assert(alce);

    return GetStringPtr(alce[nCard].dwSameAreaRuleOffset,
                        alce[nCard].dwSameAreaRuleSize);
}

inline
LPCSTR CtTranslateCaps::GetLongDistanceRule(DWORD nCard) const
{
    assert(nCard < GetNumCards());
    LPLINECARDENTRY alce = (LPLINECARDENTRY)GetDataPtr(GetData()->dwCardListOffset);
    assert(alce);

    return GetStringPtr(alce[nCard].dwLongDistanceRuleOffset,
                        alce[nCard].dwLongDistanceRuleSize);
}

inline
LPCSTR CtTranslateCaps::GetInternationalRule(DWORD nCard) const
{
    assert(nCard < GetNumCards());
    LPLINECARDENTRY alce = (LPLINECARDENTRY)GetDataPtr(GetData()->dwCardListOffset);
    assert(alce);

    return GetStringPtr(alce[nCard].dwInternationalRuleOffset,
                        alce[nCard].dwInternationalRuleSize);
}

inline
DWORD CtTranslateCaps::GetCardOptions(DWORD nCard) const
{
    assert(nCard < GetNumCards());
    LPLINECARDENTRY alce = (LPLINECARDENTRY)GetDataPtr(GetData()->dwCardListOffset);
    assert(alce);

    return alce[nCard].dwOptions;
}

#endif  // TTRANSLATECAPS_H
