// This code is a part of the Telephony Framework C++ Library.
// Copyright (C) 1997 Chris Sells. All rights reserved.
// tPhoneNo.h: CtPhoneNo class interface

/* 
This is a class to encapsulate a canonical phone number. If you've got a phone
number in it's separate parts, use:

    CtPhoneNo   pno(pszCountryCode, pszAreaCode, pszPhoneNo);

or

    CtPhoneNo   pno;
    pno.SetCanonical(pszCountryCode, pszAreaCode, pszPhoneNo);

or

    CtPhoneNo   pno;
    pno.SetCountryCode(pszCountryCode);
    pno.SetAreaCode(pszAreaCode);
    pno.SetPhoneNo(pszPhoneNo);

If you've got a phone number that may or may not be in canonical format, use:

    CtPhoneNo   pno(pszWholePhoneNo);

or

    CtPhoneNo   pno;
    pno.SetWholePhoneNo(pszPhoneNo);

To get a number back in canonical format "+CC [(AC) ]PN", use:

    MessageBox(0, pno.GetCanonical(), "The number is:", 0);

To get back the individual pieces, use:

    MessageBox(0, pno.GetCountryCode(), "The country code is:", 0);
    MessageBox(0, pno.GetAreaCode(), "The area code is:", 0);
    MessageBox(0, pno.GetPhoneNo(), "The phone number is:", 0);

All of the storage is managed by the object itself. Each of the GetXxx() functions
returns an LPCSTR to storage maintained in the object and is not guarenteed to be
the same between modifications of the CtPhoneNo object.

To get number back in displayable format "[CC ][(AC) ][PN]", use:

  MessageBox(0, pno.GetDisplayable(), "The number is:", 0);

To get the number back in a format fit for lineTranslateAddress(), use:

  ::lineTranslateAddress(..., pno.GetTranslatable(), ...);

GetTranslatable() tries for canonical and falls back on displayable.
By default, it also translates numbers that contain letters to containing only numbers.

To store a phone number for later use, serialze the result of GetTranslatable(0)
and construct w/ single argument constructor or SetWholePhoneNo().

To clear the phone number and reset the area code and country code
to the defaults for the current location, use:

    CtPhoneNo   pno;
    pno.ResetToLocation();

Note: The area code and country code defaults are set automatically at construction.
*/

#ifndef TPHONENO_H
#define TPHONENO_H

class CtPhoneNo
{
public:
	// Set default country code and area code
	CtPhoneNo();

    // Will pass szWholePhoneNo onto SetWholePhoneNo()
    CtPhoneNo(LPCSTR szWholePhoneNo);

    // These are the easy constructors
    CtPhoneNo(LPCSTR szCountryCode, LPCSTR szAreaCode, LPCSTR szPhoneNo);
    CtPhoneNo(DWORD nCountryCode, LPCSTR szAreaCode, LPCSTR szPhoneNo);

    // Copy constructor & assignment operator
    CtPhoneNo(const CtPhoneNo& pno);
    CtPhoneNo& operator=(const CtPhoneNo& pno);

    virtual ~CtPhoneNo();

    LPCSTR  GetCountryCode();
    DWORD   GetCountryCodeNum();
    LPCSTR  GetAreaCode();
    LPCSTR  GetPhoneNo();       // Just phone number part, no area code

    // Will leave letters alone.
    // Format: "+CC [(AC) ]PN"
    virtual LPCSTR  GetCanonical();

    // Will leave letters alone.
    // Format: "[CC ][(AC) ]PN"
    virtual LPCSTR  GetDisplayable();

    // Chooses canonical if available and displayable otherwise
    // (pass an alternate string to change mapping or NULL to prevent mapping).
    //                            Letters -> Digits: ABCDEFGHIJKLMNOPQRSTUVWXYZ
    virtual LPCSTR  GetTranslatable(LPCSTR pszMap = "22233344455566677778889999");

    // Given an arbitrary phone no., will split out parts if it can
    // (will test for canonical format first).
    virtual void    SetWholePhoneNo(LPCSTR szWholePhoneNo);

	// Given an phone no. in canonical format, will split it up
    // Assumes rigid format "+CC [(AC) ]PN"
	virtual void	SetCanonical(LPCSTR szCanonical);

    // Sets canonical format if everything's been broken out already
    virtual void    SetCanonical(LPCSTR szCountryCode, LPCSTR szAreaCode, LPCSTR szPhoneNo);
    virtual void    SetCanonical(DWORD nCountryCode, LPCSTR szAreaCode, LPCSTR szPhoneNo);

    // Clear phone no. and set country code and area code
    // to current location defaults
    void    ResetToLocation();

    // Set individual pieces
    void    SetCountryCode(LPCSTR szCountryCode);
    void    SetCountryCode(DWORD nCountryCode);
    void    SetAreaCode(LPCSTR szAreaCode);
    void    SetPhoneNo(LPCSTR szPhoneNo);

private:
    // Cached
    LPSTR   m_pszCountryCode;
    LPSTR   m_pszAreaCode;
    LPSTR   m_pszPhoneNo;

    // Constructed and cached
    LPSTR   m_pszCanonical;
    LPSTR   m_pszDisplayable;
    LPSTR   m_pszTranslatable;

    void    ClearConstructs();
    void    CheckDefaults();
    void    ResetAll();
    void    Copy(const CtPhoneNo& pno);
};

#endif  // TPHONENO_H
