// This code is a part of the Telephony Framework C++ Library.
// Copyright (C) 1997 Chris Sells. All rights reserved.
// tPhoneCaps.h: CtPhoneCaps struct interface

#ifndef TPHONECAPS_H
#define TPHONECAPS_H

#include "tVarData.h"

class CtPhoneCaps : public CtVariableData
{
public:
    TRESULT GetDevCaps(DWORD nPhoneID);
    
    operator const LPPHONECAPS() const;

public:
    // PHONECAPS wrappers
    LPCSTR  GetProviderInfo() const;
    LPCSTR  GetPhoneInfo() const;
    DWORD   GetPermanentPhoneID() const;
    LPCSTR  GetPhoneName() const;

    DWORD   GetPhoneStates() const;
    DWORD   GetHookSwitchDevs() const;
    DWORD   GetHandsetHookSwitchModes() const;
    DWORD   GetSpeakerHookSwitchModes() const;
    DWORD   GetHeadsetHookSwitchModes() const;
    DWORD   GetVolumeFlags() const;
    DWORD   GetGainFlags() const;

    DWORD   GetDisplayNumRows() const;
    DWORD   GetDisplayNumColumns() const;
    DWORD   GetNumRingModes() const;
    DWORD   GetNumButtonLamps() const;
    DWORD   GetButtonModes(DWORD nButton) const;
    DWORD   GetButtonFunction(DWORD nButton) const;
    DWORD   GetLampModes(DWORD nLamp) const;

    DWORD   GetNumSetData() const;
    DWORD   GetSetData(DWORD nDatum) const;
    DWORD   GetNumGetData() const;
    DWORD   GetGetData(DWORD nDatum) const;

    DWORD   GetDevSpecificSize() const;
    void*   GetDevSpecificData() const;

protected:
    virtual TRESULT FillBuffer();

private:
    DWORD   m_nPhoneID;

    const LPPHONECAPS GetData() const;
};

// inline member functions: For Compiler Eyes Only!
#include "TfxUtil.h"

inline
CtPhoneCaps::operator const LPPHONECAPS() const
{
    return LPPHONECAPS(m_pd);
}

inline
const LPPHONECAPS CtPhoneCaps::GetData() const
{
    assert(m_pd);
    return LPPHONECAPS(m_pd);
}

inline
TRESULT CtPhoneCaps::GetDevCaps(DWORD nPhoneID)
{
    assert(CtPhone::GetAppHandle());
    assert(nPhoneID < CtPhone::GetNumDevs());

    m_nPhoneID = nPhoneID;
    return UpdateData();
}

inline
TRESULT CtPhoneCaps::FillBuffer()
{
    assert(m_pd->dwTotalSize >= sizeof(PHONECAPS));
    return ::TfxTapiFunc(::phoneGetDevCaps(CtPhone::GetAppHandle(),
                                          m_nPhoneID,
                                          CtPhone::GetApiVersion(m_nPhoneID),
                                          0, LPPHONECAPS(m_pd)),
                         TDBSTR("phoneGetDevCaps"));
}

inline
LPCSTR CtPhoneCaps::GetProviderInfo() const
{
    return GetStringPtr(GetData()->dwProviderInfoOffset,
                        GetData()->dwProviderInfoSize,
                        GetData()->dwStringFormat);
}

inline
LPCSTR CtPhoneCaps::GetPhoneInfo() const
{
    return GetStringPtr(GetData()->dwPhoneInfoOffset,
                        GetData()->dwPhoneInfoSize,
                        GetData()->dwStringFormat);
}

inline
DWORD CtPhoneCaps::GetPermanentPhoneID() const
{
    return GetData()->dwPermanentPhoneID;
}

inline
LPCSTR CtPhoneCaps::GetPhoneName() const
{
    return GetStringPtr(GetData()->dwPhoneNameOffset,
                        GetData()->dwPhoneNameSize,
                        GetData()->dwStringFormat);
}

inline
DWORD CtPhoneCaps::GetPhoneStates() const
{
    return GetData()->dwPhoneStates;
}

inline
DWORD CtPhoneCaps::GetHookSwitchDevs() const
{
    return GetData()->dwHookSwitchDevs;
}

inline
DWORD CtPhoneCaps::GetHandsetHookSwitchModes() const
{
    return GetData()->dwHandsetHookSwitchModes;
}

inline
DWORD CtPhoneCaps::GetSpeakerHookSwitchModes() const
{
    return GetData()->dwSpeakerHookSwitchModes;
}

inline
DWORD CtPhoneCaps::GetHeadsetHookSwitchModes() const
{
    return GetData()->dwHeadsetHookSwitchModes;
}

inline
DWORD CtPhoneCaps::GetVolumeFlags() const
{
    return GetData()->dwVolumeFlags;
}

inline
DWORD CtPhoneCaps::GetGainFlags() const
{
    return GetData()->dwGainFlags;
}

inline
DWORD CtPhoneCaps::GetDisplayNumRows() const
{
    return GetData()->dwDisplayNumRows;
}

inline
DWORD CtPhoneCaps::GetDisplayNumColumns() const
{
    return GetData()->dwDisplayNumColumns;
}

inline
DWORD CtPhoneCaps::GetNumRingModes() const
{
    return GetData()->dwNumRingModes;
}

inline
DWORD CtPhoneCaps::GetNumButtonLamps() const
{
    return GetData()->dwNumButtonLamps;
}

inline
DWORD CtPhoneCaps::GetButtonModes(DWORD nButton) const
{
    assert(nButton < GetNumButtonLamps());

    DWORD*  rgButtonModes = (DWORD*)((BYTE*)m_pd +
                                     GetData()->dwButtonModesOffset +
                                     sizeof(DWORD) * nButton);
    return rgButtonModes[nButton];
}

inline
DWORD CtPhoneCaps::GetButtonFunction(DWORD nButton) const
{
    assert(nButton < GetNumButtonLamps());

    DWORD*  rgButtonFunctions = (DWORD*)((BYTE*)m_pd +
                                         GetData()->dwButtonFunctionsOffset +
                                         sizeof(DWORD) * nButton);
    return rgButtonFunctions[nButton];
}

inline
DWORD CtPhoneCaps::GetLampModes(DWORD nLamp) const
{
    assert(nLamp < GetNumButtonLamps());

    DWORD*  rgLampModes = (DWORD*)((BYTE*)m_pd +
                                   GetData()->dwLampModesOffset +
                                   sizeof(DWORD) * nLamp);
    return rgLampModes[nLamp];
}

inline
DWORD CtPhoneCaps::GetNumSetData() const
{
    return GetData()->dwNumSetData;
}

inline
DWORD CtPhoneCaps::GetSetData(DWORD nDatum) const
{
    assert(nDatum < GetNumSetData());

    DWORD*  rgSetData = (DWORD*)((BYTE*)m_pd +
                                 GetData()->dwSetDataOffset +
                                 sizeof(DWORD) * nDatum);
    return rgSetData[nDatum];
}

inline
DWORD CtPhoneCaps::GetNumGetData() const
{
    return GetData()->dwNumGetData;
}

inline
DWORD CtPhoneCaps::GetGetData(DWORD nDatum) const
{
    assert(nDatum < GetNumGetData());

    DWORD*  rgGetData = (DWORD*)((BYTE*)m_pd +
                                 GetData()->dwGetDataOffset +
                                 sizeof(DWORD) * nDatum);
    return rgGetData[nDatum];
}

inline
DWORD CtPhoneCaps::GetDevSpecificSize() const
{
    return GetData()->dwDevSpecificSize;
}

inline
void* CtPhoneCaps::GetDevSpecificData() const
{
    return ((BYTE*)m_pd + GetData()->dwDevSpecificOffset);
}

#endif  // TPHONECAPS_H
