// This code is a part of the Telephony Framework C++ Library.
// Copyright (C) 1997 Chris Sells. All rights reserved.
// tLineDevCaps.h: CtLineDevCaps struct interface

#ifndef TLINECAPS_H
#define TLINECAPS_H

#include "tVarData.h"

class CtLineDevCaps : public CtVariableData
{
public:
    TRESULT GetDevCaps(DWORD nLineID);
    
    operator const LPLINEDEVCAPS() const;

public:
    // LINEDEVCAPS wrappers
    LPCSTR  GetProviderInfo() const;
    LPCSTR  GetSwitchInfo() const;
    DWORD   GetPermanentLineID() const;
    LPCSTR  GetLineName() const;
    
    DWORD   GetAddressModes() const;
    DWORD   GetNumAddresses() const;
    DWORD   GetBearerModes() const;
    DWORD   GetMaxRate() const;
    DWORD   GetMediaModes() const;

    DWORD   GetGenerateToneModes() const;
    DWORD   GetGenerateToneMaxNumFreq() const;
    DWORD   GetGenerateDigitModes() const;
    DWORD   GetMonitorToneMaxNumFreq() const;
    DWORD   GetMonitorToneMaxNumEntries() const;
    DWORD   GetMonitorDigitModes() const;
    DWORD   GetGatherDigitsMinTimeout() const;
    DWORD   GetGatherDigitsMaxTimeout() const;

    DWORD   GetMedCtlDigitMaxListSize() const;
    DWORD   GetMedCtlMediaMaxListSize() const;
    DWORD   GetMedCtlToneMaxListSize() const;
    DWORD   GetMedCtlCallStateMaxListSize() const;

    DWORD   GetDevCapFlags() const;
    DWORD   GetMaxNumActiveCalls() const;
    DWORD   GetAnswerMode() const;
    DWORD   GetRingModes() const;
    DWORD   GetLineStates() const;

    DWORD   GetUUIAcceptSize() const;
    DWORD   GetUUIAnswerSize() const;
    DWORD   GetUUIMakeCallSize() const;
    DWORD   GetUUIDropSize() const;
    DWORD   GetUUISendUserUserInfoSize() const;
    DWORD   GetUUICallInfoSize() const;
    
    const LPLINEDIALPARAMS  GetMinDialParams() const;
    const LPLINEDIALPARAMS  GetMaxDialParams() const;
    const LPLINEDIALPARAMS  GetDefaultDialParams() const;

    DWORD   GetNumTerminals() const;
    LPCSTR  GetTerminalText(DWORD nTermID) const;
    const LPLINETERMCAPS    GetTermCaps(DWORD nTermID) const;

    DWORD    GetLineFeatures() const;

#if (TAPI_CURRENT_VERSION >= 0x00020000)
    DWORD   GetSettableDevStatus() const;
    DWORD   GetNumDeviceClasses() const;
    LPCSTR  GetDeviceClassName(DWORD nDeviceClass) const;
#endif

protected:
    virtual TRESULT FillBuffer();

private:
    DWORD   m_nLineID;

    const LPLINEDEVCAPS GetData() const;
};

// Inline member functions: For Compiler Eyes Only!
#include "TfxUtil.h"

inline
CtLineDevCaps::operator const LPLINEDEVCAPS() const
{
    return LPLINEDEVCAPS(m_pd);
}

inline
const LPLINEDEVCAPS CtLineDevCaps::GetData() const
{
    assert(m_pd);
    return LPLINEDEVCAPS(m_pd);
}

inline
TRESULT CtLineDevCaps::GetDevCaps(DWORD nLineID)
{
    assert(CtLine::GetAppHandle());
    assert(nLineID < CtLine::GetNumDevs());

    m_nLineID = nLineID;
    return UpdateData();
}

inline
TRESULT CtLineDevCaps::FillBuffer()
{
    assert(m_pd->dwTotalSize >= sizeof(LINEDEVCAPS));
    return ::TfxTapiFunc(::lineGetDevCaps(CtLine::GetAppHandle(),
                                          m_nLineID,
                                          CtLine::GetApiVersion(m_nLineID),
                                          0, LPLINEDEVCAPS(m_pd)),
                         TDBSTR("lineGetDevCaps"));
}

inline
LPCSTR CtLineDevCaps::GetProviderInfo() const
{
    return GetStringPtr(GetData()->dwProviderInfoOffset,
                        GetData()->dwProviderInfoSize,
                        GetData()->dwStringFormat);
}

inline
LPCSTR CtLineDevCaps::GetSwitchInfo() const
{
    return GetStringPtr(GetData()->dwSwitchInfoOffset,
                        GetData()->dwSwitchInfoSize,
                        GetData()->dwStringFormat);
}

inline
DWORD CtLineDevCaps::GetPermanentLineID() const
{
    return GetData()->dwPermanentLineID;
}

inline
LPCSTR CtLineDevCaps::GetLineName() const
{
    return GetStringPtr(GetData()->dwLineNameOffset,
                        GetData()->dwLineNameSize,
                        GetData()->dwStringFormat);
}

inline
DWORD CtLineDevCaps::GetAddressModes() const
{
    return GetData()->dwAddressModes;
}

inline
DWORD CtLineDevCaps::GetNumAddresses() const
{
    return GetData()->dwNumAddresses;
}

inline
DWORD CtLineDevCaps::GetBearerModes() const
{
    return GetData()->dwBearerModes;
}

inline
DWORD CtLineDevCaps::GetMaxRate() const
{
    return GetData()->dwMaxRate;
}

inline
DWORD CtLineDevCaps::GetMediaModes() const
{
    return GetData()->dwMediaModes;
}

inline
DWORD CtLineDevCaps::GetGenerateToneModes() const
{
    return GetData()->dwGenerateToneModes;
}

inline
DWORD CtLineDevCaps::GetGenerateToneMaxNumFreq() const
{
    return GetData()->dwGenerateToneMaxNumFreq;
}

inline
DWORD CtLineDevCaps::GetGenerateDigitModes() const
{
    return GetData()->dwGenerateDigitModes;
}

inline
DWORD CtLineDevCaps::GetMonitorToneMaxNumFreq() const
{
    return GetData()->dwMonitorToneMaxNumFreq;
}

inline
DWORD CtLineDevCaps::GetMonitorToneMaxNumEntries() const
{
    return GetData()->dwMonitorToneMaxNumEntries;
}

inline
DWORD CtLineDevCaps::GetMonitorDigitModes() const
{
    return GetData()->dwMonitorDigitModes;
}

inline
DWORD CtLineDevCaps::GetGatherDigitsMinTimeout() const
{
    return GetData()->dwGatherDigitsMinTimeout;
}

inline
DWORD CtLineDevCaps::GetGatherDigitsMaxTimeout() const
{
    return GetData()->dwGatherDigitsMaxTimeout;
}

inline
DWORD CtLineDevCaps::GetMedCtlDigitMaxListSize() const
{
    return GetData()->dwMedCtlDigitMaxListSize;
}

inline
DWORD CtLineDevCaps::GetMedCtlMediaMaxListSize() const
{
    return GetData()->dwMedCtlMediaMaxListSize;
}

inline
DWORD CtLineDevCaps::GetMedCtlToneMaxListSize() const
{
    return GetData()->dwMedCtlToneMaxListSize;
}

inline
DWORD CtLineDevCaps::GetMedCtlCallStateMaxListSize() const
{
    return GetData()->dwMedCtlCallStateMaxListSize;
}

inline
DWORD CtLineDevCaps::GetDevCapFlags() const
{
    return GetData()->dwDevCapFlags;
}

inline
DWORD CtLineDevCaps::GetMaxNumActiveCalls() const
{
    return GetData()->dwMaxNumActiveCalls;
}

inline
DWORD CtLineDevCaps::GetAnswerMode() const
{
    return GetData()->dwAnswerMode;
}

inline
DWORD CtLineDevCaps::GetRingModes() const
{
    return GetData()->dwRingModes;
}

inline
DWORD CtLineDevCaps::GetLineStates() const
{
    return GetData()->dwLineStates;
}

inline
DWORD CtLineDevCaps::GetUUIAcceptSize() const
{
    return GetData()->dwUUIAcceptSize;
}

inline
DWORD CtLineDevCaps::GetUUIAnswerSize() const
{
    return GetData()->dwUUIAnswerSize;
}

inline
DWORD CtLineDevCaps::GetUUIMakeCallSize() const
{
    return GetData()->dwUUIMakeCallSize;
}

inline
DWORD CtLineDevCaps::GetUUIDropSize() const
{
    return GetData()->dwUUIDropSize;
}

inline
DWORD CtLineDevCaps::GetUUISendUserUserInfoSize() const
{
    return GetData()->dwUUISendUserUserInfoSize;
}

inline
DWORD CtLineDevCaps::GetUUICallInfoSize() const
{
    return GetData()->dwUUICallInfoSize;
}

inline
const LPLINEDIALPARAMS CtLineDevCaps::GetMinDialParams() const
{
    return &(GetData()->MinDialParams);
}

inline
const LPLINEDIALPARAMS CtLineDevCaps::GetMaxDialParams() const
{
    return &(GetData()->MaxDialParams);
}

inline
const LPLINEDIALPARAMS CtLineDevCaps::GetDefaultDialParams() const
{
    return &(GetData()->DefaultDialParams);
}

inline
DWORD CtLineDevCaps::GetNumTerminals() const
{
    return GetData()->dwNumTerminals;
}

inline
LPCSTR CtLineDevCaps::GetTerminalText(DWORD nTermID) const
{
    assert(nTermID < GetNumTerminals());
    return GetStringPtr(GetData()->dwTerminalTextOffset +
                            GetData()->dwTerminalTextEntrySize * nTermID,
                             GetData()->dwTerminalTextEntrySize,
                             GetData()->dwStringFormat);
}

inline
const LPLINETERMCAPS CtLineDevCaps::GetTermCaps(DWORD nTermID) const
{
    assert(nTermID < GetNumTerminals());

    LPLINETERMCAPS  ptc = LPLINETERMCAPS((BYTE*)m_pd +
                                           GetData()->dwTerminalCapsOffset +
                                           sizeof(LINETERMCAPS) * nTermID);
    return ptc;
}

inline
DWORD CtLineDevCaps::GetLineFeatures() const
{
    return GetData()->dwLineFeatures;
}

#if (TAPI_CURRENT_VERSION >= 0x00020000)
inline
DWORD CtLineDevCaps::GetSettableDevStatus() const
{
    return GetData()->dwSettableDevStatus;
}

inline
DWORD CtLineDevCaps::GetNumDeviceClasses() const
{
    DWORD   n = 0;
    bool    bNull1 = false;
    LPCSTR  psz = GetStringPtr(GetData()->dwDeviceClassesOffset,
                               GetData()->dwDeviceClassesSize);

    while( true )
    {
        // Found first NULL in potential double-NULL situation
        if( !*psz && !bNull1 )
        {
            bNull1 = true;
            ++n;    // Found the end of another device class name
        }
        // Found 2nd NULL
        else if( !*psz && bNull1 )
        {
            break;
        }
        // Found !NULL
        else
        {
            bNull1 = false;
        }

        ++psz;
    }

    return n;
}

inline
LPCSTR CtLineDevCaps::GetDeviceClassName(DWORD nDeviceClass) const
{
    assert(nDeviceClass < GetNumDeviceClasses());

    bool    bNull1 = false;
    LPCSTR  psz = GetStringPtr(GetData()->dwDeviceClassesOffset,
                               GetData()->dwDeviceClassesSize);

    while( nDeviceClass )
    {
        // Found first NULL in potential double-NULL situation
        if( !*psz && !bNull1 )
        {
            bNull1 = true;
            --nDeviceClass; // Found the end of another device class name
        }
        // Found 2nd NULL
        else if( !*psz && bNull1 )
        {
            break;
        }
        // Found !NULL
        else
        {
            bNull1 = false;
        }

        ++psz;
    }

    return psz;
}

#endif
#endif  // TLINECAPS_H
