// This code is a part of the Telephony Framework C++ Library.
// Copyright (C) 1997 Chris Sells. All rights reserved.
// tDeviceID.h: CtDeviceID class interface

#ifndef TDEVICEID_H
#define TDEVICEID_H

#include "tVarData.h"

class CtDeviceID : public CtVariableData
{
public:
    operator const LPVARSTRING() const;

    // Fillers
    TRESULT GetIDFromPhone(LPCSTR szDeviceClass, HPHONE hPhone);
    TRESULT GetIDFromLine(LPCSTR szDeviceClass, HLINE hLine);
    TRESULT GetIDFromLineAddress(LPCSTR szDeviceClass, HLINE hLine, DWORD nAddressID);
    TRESULT GetIDFromCall(LPCSTR szDeviceClass, HCALL hCall);

    // Accessors
    LPCSTR  GetString();
    LPCSTR  GetHandleAndString(HANDLE* ph);
    DWORD   GetDeviceID();
    DWORD*  GetDeviceIDs();

protected:
    virtual TRESULT FillBuffer();

private:
    LPCSTR  m_pszDeviceClass;
    HPHONE  m_hPhone;
    HLINE   m_hLine;
    DWORD   m_nAddressID;
    HCALL   m_hCall;
    DWORD   m_nSelect;

    const LPVARSTRING GetData() const;
};

// Inline member functions: For Compiler Eyes Only!

#include "TfxUtil.h"

inline
CtDeviceID::operator const LPVARSTRING() const
{
    return LPVARSTRING(m_pd);
}

inline
const LPVARSTRING CtDeviceID::GetData() const
{
    assert(m_pd);
    return LPVARSTRING(m_pd);
}

inline
TRESULT CtDeviceID::GetIDFromPhone(LPCSTR szDeviceClass, HPHONE hPhone)
{
    assert(hPhone);

    m_pszDeviceClass = szDeviceClass;
    m_hPhone = hPhone;
    m_hLine = 0;
    m_nAddressID = 0;
    m_hCall = 0;
    m_nSelect = 0;

    return UpdateData();
}

inline
TRESULT CtDeviceID::GetIDFromLine(LPCSTR szDeviceClass, HLINE hLine)
{
    assert(hLine);

    m_pszDeviceClass = szDeviceClass;
    m_hPhone = 0;
    m_hLine = hLine;
    m_nAddressID = 0;
    m_hCall = 0;
    m_nSelect = LINECALLSELECT_LINE;

    return UpdateData();
}

inline
TRESULT CtDeviceID::GetIDFromLineAddress(LPCSTR szDeviceClass, HLINE hLine, DWORD nAddressID)
{
    assert(hLine);

    m_pszDeviceClass = szDeviceClass;
    m_hPhone = 0;
    m_hLine = hLine;
    m_nAddressID = nAddressID;
    m_hCall = 0;
    m_nSelect = LINECALLSELECT_ADDRESS;

    return UpdateData();
}

inline
TRESULT CtDeviceID::GetIDFromCall(LPCSTR szDeviceClass, HCALL hCall)
{
    assert(hCall);

    m_pszDeviceClass = szDeviceClass;
    m_hPhone = 0;
    m_hLine = 0;
    m_nAddressID = 0;
    m_hCall = hCall;
    m_nSelect = LINECALLSELECT_CALL;

    return UpdateData();
}

inline
TRESULT CtDeviceID::FillBuffer()
{
    assert(m_pd->dwTotalSize >= sizeof(VARSTRING));
    if( m_hPhone )
    {
        return ::TfxTapiFunc(::phoneGetID(m_hPhone,
                                          LPVARSTRING(m_pd),
                                          m_pszDeviceClass),
                             TDBSTR("phoneGetID"));
    }
    else
    {
        return ::TfxTapiFunc(::lineGetID(m_hLine, m_nAddressID, m_hCall,
                                         m_nSelect, LPVARSTRING(m_pd),
                                         m_pszDeviceClass),
                             TDBSTR("lineGetID"));
    }
}

inline
LPCSTR CtDeviceID::GetString()
{
    return GetStringPtr(GetData()->dwStringOffset,
                        GetData()->dwStringSize,
                        GetData()->dwStringFormat);
}

inline
LPCSTR CtDeviceID::GetHandleAndString(HANDLE* ph)
{
    assert(ph);
    assert(GetData()->dwStringFormat == STRINGFORMAT_BINARY);

    *ph = *(HANDLE*)GetDataPtr(sizeof(VARSTRING));
    return GetStringPtr(sizeof(VARSTRING) + sizeof(HANDLE),
                        1, STRINGFORMAT_ASCII);
}

inline
DWORD CtDeviceID::GetDeviceID()
{
    assert(GetData()->dwStringFormat == STRINGFORMAT_BINARY);

    return *(DWORD*)GetDataPtr(sizeof(VARSTRING));
}

inline
DWORD* CtDeviceID::GetDeviceIDs()
{
    assert(GetData()->dwStringFormat == STRINGFORMAT_BINARY);

    return (DWORD*)GetDataPtr(sizeof(VARSTRING));
}

#endif  // TDEVICEID_H
