// This code is a part of the Telephony Framework C++ Library.
// Copyright (C) 1997 Chris Sells. All rights reserved.
// tCallInfo.h: CtCallInfo struct interface

#ifndef TCALLINFO_H
#define TCALLINFO_H

#include "tVarData.h"

class CtCallInfo : public CtVariableData
{
public:
    TRESULT GetCallInfo(const CtCall* pCall);
    TRESULT GetCallInfo(const HCALL hCall);
    
    operator const LPLINECALLINFO() const;

public:
    // CallInfo wrappers

    HLINE   GetLineHandle() const;
    DWORD   GetLineID() const;
    DWORD   GetAddressID() const;
    DWORD   GetBearerMode() const;
    DWORD   GetRate() const;
    DWORD   GetMediaMode() const;
    DWORD   GetAppSpecific() const;

    DWORD   GetCallID() const;
    DWORD   GetRelatedCallID() const;
    DWORD   GetCallParamFlags() const;
    DWORD   GetCallStates() const;
    DWORD   GetMonitorDigitModes() const;
    DWORD   GetMonitorMediaModes() const;

    const LPLINEDIALPARAMS  GetDialParams() const;

    DWORD   GetOrigin() const;
    DWORD   GetReason() const;
    DWORD   GetCompletionID() const;

    DWORD   GetNumOwners() const;
    DWORD   GetNumMonitors() const;

    DWORD   GetCountryCode() const;
    DWORD   GetTrunk() const;

    DWORD   GetCallerIDFlags() const;
    LPCSTR  GetCallerID() const;
    LPCSTR  GetCallerIDName() const;

    DWORD   GetCalledIDFlags() const;
    LPCSTR  GetCalledID() const;
    LPCSTR  GetCalledIDName() const;

    DWORD   GetConnectedIDFlags() const;
    LPCSTR  GetConnectedID() const;
    LPCSTR  GetConnectedIDName() const;

    DWORD   GetRedirectionIDFlags() const;
    LPCSTR  GetRedirectionID() const;
    LPCSTR  GetRedirectionIDName() const;

    DWORD   GetRedirectingIDFlags() const;
    LPCSTR  GetRedirectingID() const;
    LPCSTR  GetRedirectingIDName() const;

    LPCSTR  GetAppName() const;
    LPCSTR  GetDisplayableAddress() const;
    LPCSTR  GetCalledParty() const;
    LPCSTR  GetComment() const;
    LPCSTR  GetDisplay() const;

    DWORD   GetUserUserInfoSize() const;
    void*   GetUserUserInfo() const;

    DWORD   GetHighLevelCompSize() const;
    void*   GetHighLevelComp() const;

    DWORD   GetLowLevelCompSize() const;
    void*   GetLowLevelComp() const;

    DWORD   GetChargingInfoSize() const;
    void*   GetChargingInfo() const;

    DWORD   GetNumTerminals() const;
    DWORD   GetTerminalModes(DWORD nTermID);

    DWORD   GetDevSpecificSize() const;
    void*   GetDevSpecificInfo() const;

protected:
    virtual TRESULT FillBuffer();

private:
    HCALL   m_hCall;

    const LPLINECALLINFO GetData() const;
};

// Inline member functions: For Compiler Eyes Only!
#include "TfxUtil.h"

inline
CtCallInfo::operator const LPLINECALLINFO() const
{
    return LPLINECALLINFO(m_pd);
}

inline
const LPLINECALLINFO CtCallInfo::GetData() const
{
    assert(m_pd);
    return LPLINECALLINFO(m_pd);
}

inline
TRESULT CtCallInfo::GetCallInfo(const CtCall* pCall)
{
    assert(pCall->GetHandle());

    m_hCall = pCall->GetHandle();
    return UpdateData();
}

inline
TRESULT CtCallInfo::GetCallInfo(const HCALL hCall)
{
    assert(hCall);

    m_hCall = hCall;
    return UpdateData();
}

inline
TRESULT CtCallInfo::FillBuffer()
{
    assert(m_pd->dwTotalSize >= sizeof(LINECALLINFO));
    return ::TfxTapiFunc(::lineGetCallInfo(m_hCall,
                                           LPLINECALLINFO(m_pd)),
                         TDBSTR("lineGetCallInfo"));
}

inline
HLINE CtCallInfo::GetLineHandle() const
{
    return GetData()->hLine;
}

inline
DWORD CtCallInfo::GetLineID() const
{
    return GetData()->dwLineDeviceID;
}

inline
DWORD CtCallInfo::GetAddressID() const
{
    return GetData()->dwAddressID;
}

inline
DWORD CtCallInfo::GetBearerMode() const
{
    return GetData()->dwBearerMode;
}

inline
DWORD CtCallInfo::GetRate() const
{
    return GetData()->dwRate;
}

inline
DWORD CtCallInfo::GetMediaMode() const
{
    return GetData()->dwMediaMode;
}

inline
DWORD CtCallInfo::GetAppSpecific() const
{
    return GetData()->dwAppSpecific;
}

inline
DWORD CtCallInfo::GetCallID() const
{
    return GetData()->dwCallID;
}

inline
DWORD CtCallInfo::GetRelatedCallID() const
{
    return GetData()->dwRelatedCallID;
}

inline
DWORD CtCallInfo::GetCallParamFlags() const
{
    return GetData()->dwCallParamFlags;
}

inline
DWORD CtCallInfo::GetCallStates() const
{
    return GetData()->dwCallStates;
}

inline
DWORD CtCallInfo::GetMonitorDigitModes() const
{
    return GetData()->dwMonitorDigitModes;
}

inline
DWORD CtCallInfo::GetMonitorMediaModes() const
{
    return GetData()->dwMonitorMediaModes;
}

inline
const LPLINEDIALPARAMS CtCallInfo::GetDialParams() const
{
    return &(GetData()->DialParams);
}

inline
DWORD CtCallInfo::GetOrigin() const
{
    return GetData()->dwOrigin;
}

inline
DWORD CtCallInfo::GetReason() const
{
    return GetData()->dwReason;
}

inline
DWORD CtCallInfo::GetCompletionID() const
{
    return GetData()->dwCompletionID;
}

inline
DWORD CtCallInfo::GetNumOwners() const
{
    return GetData()->dwNumOwners;
}

inline
DWORD CtCallInfo::GetNumMonitors() const
{
    return GetData()->dwNumMonitors;
}

inline
DWORD CtCallInfo::GetCountryCode() const
{
    return GetData()->dwCountryCode;
}

inline
DWORD CtCallInfo::GetTrunk() const
{
    return GetData()->dwTrunk;
}

inline
DWORD CtCallInfo::GetCallerIDFlags() const
{
    return GetData()->dwCallerIDFlags;
}

inline
LPCSTR CtCallInfo::GetCallerID() const
{
    return GetStringPtr(GetData()->dwCallerIDOffset,
                        GetData()->dwCallerIDSize);
}

inline
LPCSTR CtCallInfo::GetCallerIDName() const
{
    return GetStringPtr(GetData()->dwCallerIDNameOffset,
                        GetData()->dwCallerIDNameSize);
}

inline
DWORD CtCallInfo::GetCalledIDFlags() const
{
    return GetData()->dwCalledIDFlags;
}

inline
LPCSTR CtCallInfo::GetCalledID() const
{
    return GetStringPtr(GetData()->dwCalledIDOffset,
                        GetData()->dwCalledIDSize);
}

inline
LPCSTR CtCallInfo::GetCalledIDName() const
{
    return GetStringPtr(GetData()->dwCalledIDNameOffset,
                        GetData()->dwCalledIDNameSize);
}

inline
DWORD CtCallInfo::GetConnectedIDFlags() const
{
    return GetData()->dwConnectedIDFlags;
}

inline
LPCSTR CtCallInfo::GetConnectedID() const
{
    return GetStringPtr(GetData()->dwConnectedIDOffset,
                        GetData()->dwConnectedIDSize);
}

inline
LPCSTR CtCallInfo::GetConnectedIDName() const
{
    return GetStringPtr(GetData()->dwConnectedIDNameOffset,
                        GetData()->dwConnectedIDNameSize);
}

inline
DWORD CtCallInfo::GetRedirectionIDFlags() const
{
    return GetData()->dwRedirectionIDFlags;
}

inline
LPCSTR CtCallInfo::GetRedirectionID() const
{
    return GetStringPtr(GetData()->dwRedirectionIDOffset,
                        GetData()->dwRedirectionIDSize);
}

inline
LPCSTR CtCallInfo::GetRedirectionIDName() const
{
    return GetStringPtr(GetData()->dwRedirectionIDNameOffset,
                        GetData()->dwRedirectionIDNameSize);
}

inline
DWORD CtCallInfo::GetRedirectingIDFlags() const
{
    return GetData()->dwRedirectingIDFlags;
}

inline
LPCSTR CtCallInfo::GetRedirectingID() const
{
    return GetStringPtr(GetData()->dwRedirectingIDOffset,
                        GetData()->dwRedirectingIDSize);
}

inline
LPCSTR CtCallInfo::GetRedirectingIDName() const
{
    return GetStringPtr(GetData()->dwRedirectingIDNameOffset,
                        GetData()->dwRedirectingIDNameSize);
}

inline
LPCSTR CtCallInfo::GetAppName() const
{
    return GetStringPtr(GetData()->dwAppNameOffset,
                        GetData()->dwAppNameSize);
}

inline
LPCSTR CtCallInfo::GetDisplayableAddress() const
{
    return GetStringPtr(GetData()->dwDisplayableAddressOffset,
                        GetData()->dwDisplayableAddressSize);
}

inline
LPCSTR CtCallInfo::GetCalledParty() const
{
    return GetStringPtr(GetData()->dwCalledPartyOffset,
                        GetData()->dwCalledPartySize);
}

inline
LPCSTR CtCallInfo::GetComment() const
{
    return GetStringPtr(GetData()->dwCommentOffset,
                        GetData()->dwCommentSize);
}

inline
LPCSTR CtCallInfo::GetDisplay() const
{
    return GetStringPtr(GetData()->dwDisplayOffset,
                        GetData()->dwDisplaySize);
}

inline
DWORD CtCallInfo::GetUserUserInfoSize() const
{
    return GetData()->dwUserUserInfoSize;
}

inline
void* CtCallInfo::GetUserUserInfo() const
{
    return (void*)((BYTE*)m_pd + GetData()->dwUserUserInfoOffset);
}

inline
DWORD CtCallInfo::GetHighLevelCompSize() const
{
    return GetData()->dwHighLevelCompSize;
}

inline
void* CtCallInfo::GetHighLevelComp() const
{
    return (void*)((BYTE*)m_pd + GetData()->dwHighLevelCompOffset);
}

inline
DWORD CtCallInfo::GetLowLevelCompSize() const
{
    return GetData()->dwLowLevelCompSize;
}

inline
void* CtCallInfo::GetLowLevelComp() const
{
    return (void*)((BYTE*)m_pd + GetData()->dwLowLevelCompOffset);
}

inline
DWORD CtCallInfo::GetChargingInfoSize() const
{
    return GetData()->dwChargingInfoSize;
}

inline
void* CtCallInfo::GetChargingInfo() const
{
    return (void*)((BYTE*)m_pd + GetData()->dwChargingInfoOffset);
}

inline
DWORD CtCallInfo::GetNumTerminals() const
{
    // Save the trouble of looking up dwNumTerminals in LINEDEVCAPS
    return (GetData()->dwTerminalModesSize)/sizeof(DWORD);
}

inline
DWORD CtCallInfo::GetTerminalModes(DWORD nTermID)
{
    assert(nTermID < GetNumTerminals());

    DWORD*  adwTermModes = (DWORD*)((BYTE)m_pd + GetData()->dwTerminalModesOffset);
    assert(adwTermModes);

    return adwTermModes[nTermID];
}

inline
DWORD CtCallInfo::GetDevSpecificSize() const
{
    return GetData()->dwDevSpecificSize;
}

inline
void* CtCallInfo::GetDevSpecificInfo() const
{
    return (void*)((BYTE*)m_pd + GetData()->dwDevSpecificOffset);
}

#endif  // TCALLINFO_H
