// This code is a part of the Telephony Framework C++ Library.
// Copyright (C) 1997 Chris Sells. All rights reserved.
// tAddressCaps.h: CtAddressCaps class interface

#ifndef TADDRESSCAPS_H
#define TADDRESSCAPS_H

#include "tVarData.h"

class CtAddressCaps : public CtVariableData
{
public:
    operator const LPLINEADDRESSCAPS() const;

    TRESULT GetAddressCaps(DWORD nLineID, DWORD nAddressID);
    DWORD   GetLineDeviceID() const;
    LPCSTR  GetAddress() const;
    DWORD   GetAddressSharing() const;
    DWORD   GetAddressStates() const;
    DWORD   GetCallInfoStates() const;
    DWORD   GetCallerIDFlags() const;
    DWORD   GetCalledIDFlags() const;
    DWORD   GetConnectedIDFlags() const;
    DWORD   GetRedirectionIDFlags() const;
    DWORD   GetRedirectingIDFlags() const;
    DWORD   GetCallStates() const;
    DWORD   GetDialToneModes() const;
    DWORD   GetBusyModes() const;
    DWORD   GetSpecialInfo() const;
    DWORD   GetDisconnectModes() const;
    DWORD   GetMaxNumActiveCalls() const;
    DWORD   GetMaxNumOnHoldCalls() const;
    DWORD   GetMaxNumOnHoldPendingCalls() const;
    DWORD   GetMaxNumConference() const;
    DWORD   GetMaxNumTransConf() const;
    DWORD   GetAddrCapFlags() const;
    DWORD   GetCallFeatures() const;
    DWORD   GetRemoveFromConfCaps() const;
    DWORD   GetRemoveFromConfState() const;
    DWORD   GetTransferModes() const;
    DWORD   GetParkModes() const;
    DWORD   GetForwardModes() const;
    DWORD   GetMaxForwardEntries() const;
    DWORD   GetMaxSpecificEntries() const;
    DWORD   GetMinFwdNumRings() const;
    DWORD   GetMaxFwdNumRings() const;
    DWORD   GetMaxCallCompletions() const;
    DWORD   GetCallCompletionConds() const;
    DWORD   GetCallCompletionModes() const;
    DWORD   GetNumCompletionMessages() const;
    LPCSTR  GetCompletionMsgText(DWORD nMsg) const;
    DWORD   GetAddressFeatures() const;

#if (TAPI_CURRENT_VERSION >= 0x00020000)
    DWORD   GetPredictiveAutoTransferStates() const;
    DWORD   GetNumCallTreatments() const;
    DWORD   GetCallTreatmentID(DWORD nCallTreatment) const;
    LPCSTR  GetCallTreatmentName(DWORD nCallTreatment) const;
    DWORD   GetNumDeviceClasses() const;
    LPCSTR  GetDeviceClassName(DWORD nDeviceClass) const;
    DWORD   GetMaxCallDataSize() const;
    DWORD   GetCallFeatures2() const;
    DWORD   GetMaxNoAnswerTimeout() const;
    DWORD   GetConnectedModes() const;
    DWORD   GetOfferingModes() const;
    DWORD   GetAvailableMediaModes() const;
#endif

protected:
    virtual TRESULT FillBuffer();

private:
    DWORD   m_nLineID;
    DWORD   m_nAddressID;

    const LPLINEADDRESSCAPS GetData() const;
};

// Inline member functions: For Compiler Eyes Only!

inline
CtAddressCaps::operator const LPLINEADDRESSCAPS() const
{
    return LPLINEADDRESSCAPS(m_pd);
}

inline
const LPLINEADDRESSCAPS CtAddressCaps::GetData() const
{
    assert(m_pd);
    return LPLINEADDRESSCAPS(m_pd);
}

inline
TRESULT CtAddressCaps::GetAddressCaps(DWORD nLineID, DWORD nAddressID)
{
    assert(CtLine::GetAppHandle());
    assert(nLineID < CtLine::GetNumDevs());

    m_nLineID = nLineID;
    m_nAddressID = nAddressID;

    return UpdateData();
}

inline
TRESULT CtAddressCaps::FillBuffer()
{
    assert(m_pd->dwTotalSize >= sizeof(LINEADDRESSCAPS));
    return ::TfxTapiFunc(::lineGetAddressCaps(CtLine::GetAppHandle(),
                                              m_nLineID, m_nAddressID,
                                              CtLine::GetApiVersion(m_nLineID),
                                              0, LPLINEADDRESSCAPS(m_pd)),
                         TDBSTR("lineGetAddressCaps"));
}

inline
DWORD CtAddressCaps::GetLineDeviceID() const
{
    return GetData()->dwLineDeviceID;
}

inline
LPCSTR CtAddressCaps::GetAddress() const
{
    return GetStringPtr(GetData()->dwAddressOffset,
                        GetData()->dwAddressSize);
}

inline
DWORD CtAddressCaps::GetAddressSharing() const
{
    return GetData()->dwAddressSharing;
}

inline
DWORD CtAddressCaps::GetAddressStates() const
{
    return GetData()->dwAddressStates;
}

inline
DWORD CtAddressCaps::GetCallInfoStates() const
{
    return GetData()->dwCallInfoStates;
}

inline
DWORD CtAddressCaps::GetCallerIDFlags() const
{
    return GetData()->dwCallerIDFlags;
}

inline
DWORD CtAddressCaps::GetCalledIDFlags() const
{
    return GetData()->dwCalledIDFlags;
}

inline
DWORD CtAddressCaps::GetConnectedIDFlags() const
{
    return GetData()->dwConnectedIDFlags;
}

inline
DWORD CtAddressCaps::GetRedirectionIDFlags() const
{
    return GetData()->dwRedirectionIDFlags;
}

inline
DWORD CtAddressCaps::GetRedirectingIDFlags() const
{
    return GetData()->dwRedirectingIDFlags;
}

inline
DWORD CtAddressCaps::GetCallStates() const
{
    return GetData()->dwCallStates;
}

inline
DWORD CtAddressCaps::GetDialToneModes() const
{
    return GetData()->dwDialToneModes;
}

inline
DWORD CtAddressCaps::GetBusyModes() const
{
    return GetData()->dwBusyModes;
}

inline
DWORD CtAddressCaps::GetSpecialInfo() const
{
    return GetData()->dwSpecialInfo;
}

inline
DWORD CtAddressCaps::GetDisconnectModes() const
{
    return GetData()->dwDisconnectModes;
}

inline
DWORD CtAddressCaps::GetMaxNumActiveCalls() const
{
    return GetData()->dwMaxNumActiveCalls;
}

inline
DWORD CtAddressCaps::GetMaxNumOnHoldCalls() const
{
    return GetData()->dwMaxNumOnHoldCalls;
}

inline
DWORD CtAddressCaps::GetMaxNumOnHoldPendingCalls() const
{
    return GetData()->dwMaxNumOnHoldPendingCalls;
}

inline
DWORD CtAddressCaps::GetMaxNumConference() const
{
    return GetData()->dwMaxNumConference;
}

inline
DWORD CtAddressCaps::GetMaxNumTransConf() const
{
    return GetData()->dwMaxNumTransConf;
}

inline
DWORD CtAddressCaps::GetAddrCapFlags() const
{
    return GetData()->dwAddrCapFlags;
}

inline
DWORD CtAddressCaps::GetCallFeatures() const
{
    return GetData()->dwCallFeatures;
}

inline
DWORD CtAddressCaps::GetRemoveFromConfCaps() const
{
    return GetData()->dwRemoveFromConfCaps;
}

inline
DWORD CtAddressCaps::GetRemoveFromConfState() const
{
    return GetData()->dwRemoveFromConfState;
}

inline
DWORD CtAddressCaps::GetTransferModes() const
{
    return GetData()->dwTransferModes;
}

inline
DWORD CtAddressCaps::GetParkModes() const
{
    return GetData()->dwParkModes;
}

inline
DWORD CtAddressCaps::GetForwardModes() const
{
    return GetData()->dwForwardModes;
}

inline
DWORD CtAddressCaps::GetMaxForwardEntries() const
{
    return GetData()->dwMaxForwardEntries;
}

inline
DWORD CtAddressCaps::GetMaxSpecificEntries() const
{
    return GetData()->dwMaxSpecificEntries;
}

inline
DWORD CtAddressCaps::GetMinFwdNumRings() const
{
    return GetData()->dwMinFwdNumRings;
}

inline
DWORD CtAddressCaps::GetMaxFwdNumRings() const
{
    return GetData()->dwMaxFwdNumRings;
}

inline
DWORD CtAddressCaps::GetMaxCallCompletions() const
{
    return GetData()->dwMaxCallCompletions;
}

inline
DWORD CtAddressCaps::GetCallCompletionConds() const
{
    return GetData()->dwCallCompletionConds;
}

inline
DWORD CtAddressCaps::GetCallCompletionModes() const
{
    return GetData()->dwCallCompletionModes;
}

inline
DWORD CtAddressCaps::GetNumCompletionMessages() const
{
    return GetData()->dwNumCompletionMessages;
}

inline
LPCSTR CtAddressCaps::GetCompletionMsgText(DWORD nMsg) const
{
    assert(nMsg < GetNumCompletionMessages());
    return GetStringPtr(GetData()->dwCompletionMsgTextOffset +
                            GetData()->dwCompletionMsgTextEntrySize * nMsg,
                        GetData()->dwCompletionMsgTextEntrySize);
}

inline
DWORD CtAddressCaps::GetAddressFeatures() const
{
    return GetData()->dwAddressFeatures;
}

#if (TAPI_CURRENT_VERSION >= 0x00020000)
inline
DWORD CtAddressCaps::GetPredictiveAutoTransferStates() const
{
    return GetData()->dwPredictiveAutoTransferStates;
}

inline
DWORD CtAddressCaps::GetNumCallTreatments() const
{
    return GetData()->dwNumCallTreatments;
}

inline
DWORD CtAddressCaps::GetCallTreatmentID(DWORD nCallTreatment) const
{
    assert(nCallTreatment < GetNumCallTreatments());

    LPLINECALLTREATMENTENTRY    pcte = LPLINECALLTREATMENTENTRY((BYTE*)m_pd +
                                                                GetData()->dwCallTreatmentListOffset +
                                                                sizeof(LINECALLTREATMENTENTRY) * nCallTreatment);
    return pcte->dwCallTreatmentID;
}

inline
LPCSTR CtAddressCaps::GetCallTreatmentName(DWORD nCallTreatment) const
{
    assert(nCallTreatment < GetNumCallTreatments());

    LPLINECALLTREATMENTENTRY    pcte = LPLINECALLTREATMENTENTRY((BYTE*)m_pd +
                                                                GetData()->dwCallTreatmentListOffset +
                                                                sizeof(LINECALLTREATMENTENTRY) * nCallTreatment);
    return GetStringPtr(pcte->dwCallTreatmentNameOffset,
                        pcte->dwCallTreatmentNameSize);
}

inline
DWORD CtAddressCaps::GetNumDeviceClasses() const
{
    DWORD   n = 0;
    bool    bNull1 = false;
    LPCSTR  psz = GetStringPtr(GetData()->dwDeviceClassesOffset,
                               GetData()->dwDeviceClassesSize);

    while( true )
    {
        // Found first NULL in potential double-NULL situation
        if( !*psz && !bNull1 )
        {
            bNull1 = true;
            ++n;    // Found the end of another device class name
        }
        // Found 2nd NULL
        else if( !*psz && bNull1 )
        {
            break;
        }
        // Found !NULL
        else
        {
            bNull1 = false;
        }

        ++psz;
    }

    return n;
}

inline
LPCSTR CtAddressCaps::GetDeviceClassName(DWORD nDeviceClass) const
{
    assert(nDeviceClass < GetNumDeviceClasses());

    bool    bNull1 = false;
    LPCSTR  psz = GetStringPtr(GetData()->dwDeviceClassesOffset,
                               GetData()->dwDeviceClassesSize);

    while( nDeviceClass )
    {
        // Found first NULL in potential double-NULL situation
        if( !*psz && !bNull1 )
        {
            bNull1 = true;
            --nDeviceClass; // Found the end of another device class name
        }
        // Found 2nd NULL
        else if( !*psz && bNull1 )
        {
            break;
        }
        // Found !NULL
        else
        {
            bNull1 = false;
        }

        ++psz;
    }

    return psz;
}

inline
DWORD CtAddressCaps::GetMaxCallDataSize() const
{
    return GetData()->dwMaxCallDataSize;
}

inline
DWORD CtAddressCaps::GetCallFeatures2() const
{
    return GetData()->dwCallFeatures2;
}

inline
DWORD CtAddressCaps::GetMaxNoAnswerTimeout() const
{
    return GetData()->dwMaxNoAnswerTimeout;
}

inline
DWORD CtAddressCaps::GetConnectedModes() const
{
    return GetData()->dwConnectedModes;
}

inline
DWORD CtAddressCaps::GetOfferingModes() const
{
    return GetData()->dwOfferingModes;
}

inline
DWORD CtAddressCaps::GetAvailableMediaModes() const
{
    return GetData()->dwAvailableMediaModes;
}

#endif  // TAPI_CURRENT_VERSION >= 0x00020000
#endif  // TADDRESSCAPS_H
